"""
I have two parallel arrays. One array of dataframes. One array of titles. Each dataframe's index contains a "Date". The remaining dataframe columns show security prices, one list of prices per column. I'd like a function that takes these arrays and generates an html page of interactive plotly graphs (one graph per data frame) each using the associated title from the title array. I would also like to pass the function a "Page Header" argument that contains some preface information in html format. Please write this function for me.

"""
from typing import List, Optional, Literal
import pandas as pd
import plotly.graph_objects as go
from plotly.io import to_html
import html
import uuid

def generate_plotly_page_from_index(
    dataframes: List[pd.DataFrame],
    titles: List[str],
    page_header_html: str,
    filename: Optional[str] = None,
    document_title: str = "Interactive Charts",
    yaxis_label: str = "Price",
    legend_side: Literal["left", "right"] = "right",
) -> str:
    """
    Build an HTML page with one interactive Plotly chart per DataFrame.
    - Assumes each DataFrame has a Date-like index.
    - Clickable labels (legend items for toggling series) are placed vertically
      along the y-axis side (left or right).

    Parameters
    ----------
    dataframes : List[pd.DataFrame]
        Each must have a datetime (or coercible) index; columns are series to plot.
    titles : List[str]
        One title per DataFrame.
    page_header_html : str
        HTML preface injected near the top of the page (not sanitized).
    filename : Optional[str]
        If provided, writes the full HTML to this path.
    document_title : str
        <title> element and top H1 text.
    yaxis_label : str
        Label for the Y axis on each chart.
    legend_side : {"left","right"}
        Side of the y-axis where the clickable legend labels sit.

    Returns
    -------
    str
        The complete HTML document.
    """
    if len(dataframes) != len(titles):
        raise ValueError("`dataframes` and `titles` must be the same length.")

    # Legend anchored alongside the y-axis
    if legend_side == "left":
        legend_x = -0.1     # slightly outside plot to the left
        legend_xanchor = "right"
    else:  # "right"
        legend_x = 1.02     # slightly outside plot to the right
        legend_xanchor = "left"

    sections = []
    for i, (df, title) in enumerate(zip(dataframes, titles), start=1):
        if df.index is None or len(df.index) == 0:
            raise ValueError(f"DataFrame #{i} has an empty index.")

        work = df.copy()

        # Ensure datetime index
        if not pd.api.types.is_datetime64_any_dtype(work.index):
            try:
                work.index = pd.to_datetime(work.index, errors="coerce")
            except Exception as e:
                raise ValueError(f"DataFrame #{i} index cannot be parsed as dates: {e}")
        work = work[~work.index.isna()].sort_index()

        # Only plot numeric columns
        value_cols = [c for c in work.columns if pd.api.types.is_numeric_dtype(work[c])]
        if not value_cols:
            raise ValueError(f"DataFrame #{i} has no numeric columns to plot.")

        fig = go.Figure()
        for col in value_cols:
            fig.add_trace(go.Scatter(
                x=work.index,
                y=work[col],
                mode="lines",
                name=str(col),
                hovertemplate="%{x|%Y-%m-%d}<br>%{fullData.name}: %{y}<extra></extra>",
            ))

        fig.update_layout(
            title=title,
            margin=dict(l=60, r=60, t=60, b=40),
            # Clickable labels (legend) placed vertically alongside the y-axis
            legend=dict(
                orientation="v",
                x=legend_x, xanchor=legend_xanchor,
                y=0.5, yanchor="middle",
                bgcolor="rgba(255,255,255,0.6)",
                bordercolor="rgba(0,0,0,0.08)", borderwidth=1
            ),
            xaxis=dict(
                title="Date",
                rangeslider=dict(visible=True),
                rangeselector=dict(
                    buttons=[
                        dict(count=1, label="1m", step="month", stepmode="backward"),
                        dict(count=3, label="3m", step="month", stepmode="backward"),
                        dict(count=6, label="6m", step="month", stepmode="backward"),
                        dict(count=1, label="YTD", step="year", stepmode="todate"),
                        dict(count=1, label="1y", step="year", stepmode="backward"),
                        dict(step="all", label="All")
                    ]
                ),
                type="date"
            ),
            yaxis=dict(title=yaxis_label),
        )

        # Unique container ID per chart
        section_id = f"chart-{uuid.uuid4().hex[:8]}"
        snippet = to_html(
            fig,
            include_plotlyjs=False,   # loaded once globally
            full_html=False,
            config=dict(displayModeBar=True, responsive=True),
            div_id=section_id
        )

        sections.append(f"""
        <section class="chart-section">
          <h2>{html.escape(title)}</h2>
          {snippet}
        </section>
        """)

    html_doc = f"""<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title>{html.escape(document_title)}</title>
<script src="https://cdn.plot.ly/plotly-2.35.2.min.js"></script>
<style>
  :root {{
    --maxw: 1200px;
  }}
  body {{
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Inter, "Helvetica Neue", Arial, "Noto Sans";
    margin: 0;
    padding: 0;
    background: #fafafa;
    color: #111;
    line-height: 1.45;
  }}
  header {{
    background: #fff;
    border-bottom: 1px solid #e6e6e6;
    padding: 24px 16px;
  }}
  .container {{
    max-width: var(--maxw);
    margin: 0 auto;
    padding: 0 12px;
  }}
  .preface, .chart-section {{
    background: #ffffff;
    border: 1px solid #ececec;
    border-radius: 12px;
    padding: 16px;
    margin: 18px 0;
    box-shadow: 0 1px 2px rgba(0,0,0,0.03);
  }}
  h1, h2 {{
    margin: 0.2em 0 0.6em 0;
    font-weight: 650;
  }}
  h1 {{ font-size: 1.6rem; }}
  h2 {{ font-size: 1.25rem; }}
  footer {{
    text-align: center;
    color: #666;
    font-size: 0.9rem;
    padding: 24px 0 40px;
  }}
</style>
</head>
<body>
  <header>
    <div class="container">
      <h1>{html.escape(document_title)}</h1>
    </div>
  </header>

  <main class="container">
    <div class="preface">
      {page_header_html}
    </div>
    {''.join(sections)}
  </main>

  <footer>Generated with Plotly</footer>
</body>
</html>"""

    if filename:
        with open(filename, "w", encoding="utf-8") as f:
            f.write(html_doc)

    return html_doc

