﻿"""
https://chatgpt.com/share/68f19ead-a010-8000-9fac-e73ab254fa94

Here is a spreadsheet (from 20251002 David Brodwin's presentation).
Three equity price histories are entered in columns B, E and H (labeled “EC”) for equites S398, SB029 and SB041 respectively.
Monthly return for each equity is calculated in columns C, F and I (labeled “Mo Return”) respectively.
Maximum Draw down is calculated in columns D, G and J (labeled “DD%”) respectively.
CAGR, Mean Return, Max Draw Down and MAR are saluted for each equity in rows 29,30,31 and 32 respectively.

The goal is to construct a “Composite Portfolio” with the “EC” in K, the “Mo return” in L and the “DD%” in M.

This spreadsheet is passed to Excel’s “Solver”.
With the Objective set to maximize $L$32 (The composite portfolio’s MAR) by changing the values in P2,Q2 and R2
subject to the constraints
$P$2 <=1
$Q$2 <=1
$R$2 <=1
$S$2 <=1
Using a GRG Nonlinear solver method.
This solver is used for problems that are smooth nonlinear.
Please provide a python program that does the same thing.
The program should be passed two argument:
a data frame containing dates in the first column, Equity names in the column headers, and equity price histories in the remaining columns.
Use whatever python optimizer package makes the most sense (CVXPY, PyPortfolioOpt or scipy.optimize.minimize)

Please regenerate after adding a "MaxDD" argument to optimize_mar_from_prices.  E.g. setting MaxDD to "10" will maximize CAGR subject to MaxDD<=10%
20220111PlotReturnVolatilty
https://stocksofinterest.com/
"""
header="""
<p>
I applied David Brodwin's "Using Excel for Optimizing Blended Strategies" approach
<a href="https://paseman.com/Posts/20251021%20Blending%20Strategies/Blending%20Strategies%20with%20Excel%20part%202.pdf">pdf</a>
to the strategy lists David, Homer, Matt, PaulP, Ren, and Scott sent me.
(David (D) for example uses 'S398','SB029','SB041'.).
Instead of Excel, I used python's scipy.optimize.minimize routine (as per David's (and Don's) suggestion).
I ran it for several different timeframes, with and without the MaxDD <= 10 constraint
The results summary is in the table below.
Note that I get the same results in python as David got in Excel [0.02, 0.72, 0.27].
But note how much the allocation skews when I extend an extra 8 months.
Ren's strategy lists give a mar of 15 for the 2022-12-30,2024-12-31 case,
but the 2022-12-30,2025-08-29 timeframe did not converge (error was "Optimization failed to converge for all initializations"),
so I put zeros for the value.
Below the table are the Equity curves in plotly for all the cases.
Since it is plotly, you can click on the labels to make the curve disappear.
</p>
"""

# pip install numpy pandas scipy plotly kaleido
import pandas as pd
pd.set_option('display.max_rows', None)
pd.set_option('display.max_columns', 100)
pd.set_option('display.width', 350)
from time import time, localtime
import glob  
from mar_optimizer import optimize_mar_from_prices,_drawdown_stats,_returns_from_prices
from generate_plotly_page import generate_plotly_page_from_index

# optimize_mar_from_prices Notes:
# - You can pass MaxDD as 10 (percent) or 0.10 (decimal). Both mean 10%.
# - Constraints enforced: 0 <= w_i <= wmax, sum(w_i) == 1 (no leverage).
# - Solver: SLSQP with multi-starts; objective switches based on MaxDD being set.
# If you want the drawdown series too:
# from mar_optimizer import _drawdown_stats
# _, dd_series, _ = _drawdown_stats(comp_returns)


def createReturnsDF(df,result):#, title):
  rdf=_returns_from_prices(df)
  ys=["comp_ec"]
  for columnName in rdf.columns:
    y=columnName+"_ec"
    ys.append(y)
    rdf[y]=(1 + rdf[columnName]).cumprod()
  rdf["comp_ec"]=result.composite_curve          # equity curve (EC)
  return rdf[ys]

# S  2025 > 250829 > Strats0  > Final > 20250830.1450_ECs_20240611_Strats0-all-mar_19941230-20250829.xlsx
# SB 2025 > 250829 > Strats7  > Final > 20250830.1450_ECs_20240611_Strats7-all-mar_19941230-20250829.xlsx
# SD 2025 > 250829 > Strats9  > Final > 20250830.1440_ECs_20241231_Strats9-all-mar_19941230-20250829.xlsx
# SE 2025 > 250829 > Strats10 > Final > 20250830.1427_ECs_20241231_Strats10-all-mar_19941230-20250829.xlsx
def get_column(stratName,dfs): #Only returns the first one (e.g. S398) encountered
  for df in dfs:
    for columnName in df.columns:
      if stratName in columnName:
        return df[columnName]

def createStrategyDF(stratNames,begDate,endDate,path="Strats/*.xlsx"):
  dfs= [pd.read_excel(filename,index_col=0) for filename in glob.glob(path)]
  dfs.reverse() # Put Strats0 last.
  df=pd.DataFrame()
  for stratName in stratNames:
    column=get_column(stratName,dfs)
    df[stratName]= column[begDate:endDate]
  return df

def runOptimizer(Ss,keys,begEnds):
  titles=[]
  rdfs=[]
  rows=[]
  columnNames=["Name","Begin","End","Portfolio","Optimal Weights","Mean Return","CAGR","MaxDD","MAR",]
  for key in keys:
    for beg,end in begEnds:
      df=createStrategyDF(Ss[key],beg,end)
      try:
        for MaxDD in [None,10]:
          result = optimize_mar_from_prices(df, wmax=1.00, n_restarts=25, MaxDD=MaxDD)
          if MaxDD is None:
            titles.append("case- %s beg %s end %s maximize CAGR, wmax=1.00"%(key,beg,end))
          else:
            titles.append("case- %s beg %s end %s maximize CAGR, MaxDD <= %d%%, wmax=1.00"%(key,beg,end,MaxDD))
          rdfs.append(createReturnsDF(df,result))
          weights = [f"{weight:,.2f}" for weight in result.weights]
          rows.append([key,beg,end,list(df.columns),weights,
                       result.mean_return,result.cagr,result.max_drawdown, result.mar])
      except Exception as e:
        rows.append([key,beg,end,list(df.columns),0,0,0,0, 0])
        print("****Case failed: ", e)
  return rdfs,titles,rows,columnNames


def main():
  Ss={
  'David':['S398','SB029','SB041'],
  'Homer':['SB014', 'SB016', 'SB017', 'SB018', 'SB047'],
  'Matt' :['SE034', 'SE139', 'SE223', 'S398'],
  'PaulP':['SB016', 'SB068'],
  'Ren'  :['SB016', 'SD032', 'SD035', 'SD175'],
  'RichR':['SE007', 'SB067', 'SE033'],
  'Scott':['S398', 'SB013', 'SB023', 'SB047', 'SB061', 'SB081']}
  #J nasdaq 100 top 4, with the absolute momentum timer
  begends=[("2022-12-30","2024-12-31"),("2022-12-30","2025-08-29")]

  start_time = time()
  rdfs,titles,rows,columnNames =runOptimizer(Ss,Ss.keys(),begends)#,("2014-12-30","2025-08-29")  ['Ren'],[("2022-12-30","2024-12-31")])#
  t=time() - start_time
  print("calculate %2.6f seconds"%t)
  df=pd.DataFrame(data=rows,columns=columnNames)
  #styled_df=df.style.set_properties( subset=['Portfolio'], **{'max-width': '150px'} )
  html=header+df.to_html()

  generate_plotly_page_from_index(rdfs,titles,html,"index.html",
                                  document_title="CIMI Favorite Strategies",yaxis_label="Return")

main()


